/*******************************************************************************
* GAUSS code: Example 6.8, Table 6.2 first part 
* file: Estimate_GARCH_type_Models_SETAR_new.GS
*
* This is the GAUSS code for estimating the SETAR and SETAR-GARCH-type models
* The output serves as input for the GAUSS code "Parametric_Tests_new.GS". 
*
% Written by: Yi-Ting Chen, Academia Sinica
*
* Reference:
* Chen, Y.-T. (2008).
*    A unified approach to standardized-residuals-based correlation tests for
*    GARCH-type models. Journal of Applied Econometrics, 23(1), 111-133.
*    DOI: 10.1002/jae.985.
********************************************************************************/

library pgraph,cml;
#include cml.ext;
graphset;
output file=c:\...\likelihood.result reset; 
output off;

@******************************@
@  DEFINE STOCK INDEX RETURNS  @
@******************************@
/* The daily HSI index for the year 2010 */  
load HSI[253,1] = c:\...\HSI-daily-stockprices-2010-T253.dat; @ HSI @ 
rHSI            = 100*(ln(HSI[2:253])-ln(HSI[1:252]));  

DGP = 1;
do while DGP<=1;
  if DGP==1; y=rHSI; yt="HSI "; endif;
  _cml_Algorithm = 1;   @ The BFGS(Broyden, Fletcher, Goldfarb, and Shanno) method @ 
  _cml_MaxIters  = 50;
  _cml_DirTol    = 0.00001;
  _cml_Gradstep  = 0.0001;

  output off; 
  @*****************************@
  @  ESTIMATE AR(1)-GARCH(1,1)  @
  @*****************************@
  _cml_C = {0  0  1  0  0, 
            0  0  0  1  0, 
            0  0  0  0  1, 
            0  0  0 -1 -1}; 
  _cml_D = {0, 0, 0, -1}; 
  b0     = 0|0|0.01|0.9|0.05; 

  {QMLE1,fhat1,grad1,cov1,ret1} = CML(y,0,&AR_GARCH,b0);

  @*****************************@
  @  ESTIMATE AR(1)-EGARCH(1,1) @
  @*****************************@
  _cml_C = {0  0  0  0  0 0}; 
  _cml_D = {0}; 
  b0     = 0|0|-0.1|0.9|-0.1|0.05;  

  {QMLE2,fhat2,grad2,cov2,ret2} = CML(y,0,&AR_EGARCH,b0);

  @*********************************@
  @  ESTIMATE SETAR(1,1)-GARCH(1,1) @
  @*********************************@
  b0     = 0|0|0|0|0|0.01|0.9|0.05; 
  _cml_C = {0  0  0  0  0  1  0  0, 
            0  0  0  0  0  0  1  0, 
            0  0  0  0  0  0  0  1, 
            0  0  0  0  0  0 -1 -1}; 
  _cml_D = {0, 0, 0, -1}; 

  {QMLE3,fhat3,grad3,cov3,ret3} = CML(y,0,&SETAR_GARCH,b0);

  @**********************************@
  @  ESTIMATE SETAR(1,1)-EGARCH(1,1) @
  @**********************************@
  _cml_C = {0  0  0  0  0 0 0 0 0}; 
  _cml_D = {0}; 
  b0     = 0|0|0|0|0|-0.1|0.9|-0.1|0.05;  

  {QMLE4,fhat4,grad4,cov4,ret4} = CML(y,0,&SETAR_EGARCH,b0);
  
  @*************************@
  @  ESTIMATE GARCH(1,1)-M  @
  @*************************@
  _cml_C = {0  0 0  1  0  0, 
            0 0  0  0  1  0, 
            0 0  0  0  0  1, 
            0 0  0  0 -1 -1}; 
  _cml_D = {0, 0, 0, -1}; 
  b0     = 0|0|0|0.01|0.9|0.05;  

  {QMLE5,fhat5,grad5,cov5,ret5} = CML(y,0,&GARCH_M,b0);

  @*************************@
  @  ESTIMATE EGARCH(1,1)-M @
  @*************************@
  _cml_C = {0  0  0  0  0 0  0}; 
  _cml_D = {0}; 
  /* b0=0|0|0|-0.1|0.9|-0.1|0.05; */

  b0 = 0.0396|0.0430|-0.0062|-0.1773|0.6752|-0.1411|0.3607;

  {QMLE6,fhat6,grad6,cov6,ret6} = CML(y,0,&EGARCH_M,b0);

  @**********************************@
  @  ESTIMATE SETAR(1,1)             @
  @**********************************@
  _cml_C = {0  0  0  0  0}; 
  _cml_D = {0}; 
  b0     = 0|0|0|0|0;  

  {QMLE7,fhat7,grad7,cov7,ret7} = CML(y,0,&SETAR,b0);

  output on;
  FORMAT /RD 6, 5;

  print yt "     AR-GARCH: " QMLE1';
  print yt "     AR-EGARCH:" QMLE2';
  print yt "  SETAR-GARCH: " QMLE3';
  print yt "  SETAR-EGARCH:" QMLE4';
  print yt "  GARCH-M:     " QMLE5';
  print yt " EGARCH-M:     " QMLE6';
  print yt " SETAR:        " QMLE7';

  print yt "     AR-GARCH: " fhat1';
  print yt "     AR-EGARCH:" fhat2';
  print yt "  SETAR-GARCH: " fhat3';
  print yt "  SETAR-EGARCH:" fhat4';
  print yt "  GARCH-M:     " fhat5';
  print yt " EGARCH-M:     " fhat6';
  print yt " SETAR   :     " fhat7';

  DGP=DGP+1;
endo;

@***********************************************@
@  THE GAUSSIAN QUASI-LOG-LIKELIHOOD FUNCTIONS  @
@***********************************************@

/**********************************/
/* I. THE AR(1)-GARCH(1,1) MODEL  */
/**********************************/
proc AR_GARCH(b,y);            
local T,h,u,m,i;
T = rows(y);
h = zeros(T,1);
u = zeros(T,1);
m = zeros(T,1);

h[1] = stdc(y)^2;
u[1] = y[1]-meanc(y);

i = 2;
do while i<=T;
  m[i] = b[1]+b[2]*y[i-1];       
  h[i] = b[3]+b[4]*h[i-1]+b[5]*u[i-1]^2;
  u[i] = y[i]-m[i];
  i    = i+1;
endo;

retp(-0.5*ln(2*pi)-0.5*ln(h)-0.5*((u.^2)./h));
endp;

/************************************/
/* II. THE AR(1)-EGARCH(1,1) MODEL  */
/************************************/
proc AR_EGARCH(b,y);            
local T,h,u,m,v,av,lambda,i;
T  = rows(y);
h  = zeros(T,1);
u  = zeros(T,1);
m  = zeros(T,1);
v  = zeros(T,1);
av = zeros(T,1);

lambda = 10^(-3);
h[1]   = stdc(y)^2;
u[1]   = y[1]-meanc(y);
v[1]   = u[1]/h[1]^0.5;
av[1]  = (lambda^2+v[1]^2)^0.5;

i = 2;
do while i<=T;
  m[i]  = b[1]+b[2]*y[i-1];       
  h[i]  = exp(b[3]+b[4]*ln(h[i-1])+b[5]*(u[i-1]/h[i-1]^0.5)+b[6]*av[i-1]);
  u[i]  = y[i]-m[i];
  v[i]  = u[i]/h[i]^0.5;
  av[i] = (lambda^2+v[i]^2)^0.5;
  i     = i+1;
endo;

retp(-0.5*ln(2*pi)-0.5*ln(h)-0.5*((u.^2)./h));
endp;

/***************************************/
/* III. THE SETAR(1)-GARCH(1,1) MODEL  */
/***************************************/
proc SETAR_GARCH(b,y);            
local T,h,u,m,i,lambda,Tran;
T = rows(y);
h = zeros(T,1);
u = zeros(T,1);
m = zeros(T,1);

lambda = 10^(-3);
h[1]   = stdc(y)^2;
u[1]   = y[1]-meanc(y);

i = 2;
do while i<=T;
  Tran = 1/(1+exp((y[i-1]-b[5])/lambda));
  m[i] = Tran*(b[1]+b[2]*y[i-1])+(1-Tran)*(b[3]+b[4]*y[i-1]);       
  h[i] = b[6]+b[7]*h[i-1]+b[8]*u[i-1]^2;
  u[i] = y[i]-m[i];
  i    = i+1;
endo;

retp(-0.5*ln(2*pi)-0.5*ln(h)-0.5*((u.^2)./h));
endp;

/***************************************/
/* IV. THE SETAR(1)-EGARCH(1,1) MODEL  */
/***************************************/
proc SETAR_EGARCH(b,y);            
local T,h,u,m,lambda,v,av,i,Tran;
T = rows(y);
h = zeros(T,1);
u = zeros(T,1);
m = zeros(T,1);

lambda = 10^(-3);
v      = zeros(T,1);
av     = zeros(T,1);
h[1]   = stdc(y)^2;
u[1]   = y[1]-meanc(y);
v[1]   = u[1]/h[1]^0.5;
av[1]  = (lambda^2+v[1]^2)^0.5;

i = 2;
do while i<=T;
  Tran  = 1/(1+exp((y[i-1]-b[5])/lambda));
  m[i]  = Tran*(b[1]+b[2]*y[i-1])+(1-Tran)*(b[3]+b[4]*y[i-1]);       
  h[i]  = exp(b[6]+b[7]*ln(h[i-1])+b[8]*(u[i-1]/h[i-1]^0.5)+b[9]*av[i-1]);
  u[i]  = y[i]-m[i];
  v[i]  = u[i]/h[i]^0.5;
  av[i] = (lambda^2+v[i]^2)^0.5;
  i     = i+1;
endo;

retp(-0.5*ln(2*pi)-0.5*ln(h)-0.5*((u.^2)./h));
endp;

/******************************/
/* V. THE GARCH(1,1)-M MODEL  */
/******************************/
proc GARCH_M(b,y);            
local T,h,u,m,i;
T = rows(y);
h = zeros(T,1);
u = zeros(T,1);
m = zeros(T,1);

h[1] = stdc(y)^2;
u[1] = y[1]-meanc(y);

i = 2;
do while i<=T;
  h[i] = b[4]+b[5]*h[i-1]+b[6]*u[i-1]^2;
  m[i] = b[1]+b[2]*y[i-1]+b[3]*ln(h[i]);
  u[i] = y[i]-m[i];
  i    = i+1;
endo;

retp(-0.5*ln(2*pi)-0.5*ln(h)-0.5*((u.^2)./h));
endp;

/********************************/
/* VI. THE EGARCH(1,1)-M MODEL  */
/********************************/
proc EGARCH_M(b,y);            
local T,h,u,m,v,av,lambda,i;
T  = rows(y);
h  = zeros(T,1);
u  = zeros(T,1);
m  = zeros(T,1);
v  = zeros(T,1);
av = zeros(T,1);

lambda = 10^(-3);
h[1]   = stdc(y)^2;
u[1]   = y[1]-meanc(y);
v[1]   = u[1]/h[1]^0.5;
av[1]  = (lambda^2+v[1]^2)^0.5;

i = 2;
do while i<=T;
  h[i]  = exp(b[4]+b[5]*ln(h[i-1])+b[6]*(u[i-1]/h[i-1]^0.5)+b[7]*av[i-1]);
  m[i]  = b[1]+b[2]*y[i-1]+b[3]*ln(h[i]);
  u[i]  = y[i]-m[i];
  v[i]  = u[i]/h[i]^0.5;
  av[i] = (lambda^2+v[i]^2)^0.5;
  i     = i+1;
endo;

retp(-0.5*ln(2*pi)-0.5*ln(h)-0.5*((u.^2)./h));
endp;


/***************************************/
/* VII. THE SETAR(1)            MODEL  */
/***************************************/
proc SETAR(b,y);            
local T,h,u,m,lambda,v,av,i,Tran;
T = rows(y);
u = zeros(T,1);
m = zeros(T,1);

lambda = 10^(-3);
v      = zeros(T,1);
av     = zeros(T,1);
u[1]   = y[1]-meanc(y);
av[1]  = (lambda^2)^0.5;

i = 2;
do while i<=T;
  Tran  = 1/(1+exp((y[i-1]-b[5])/lambda));
  m[i]  = Tran*(b[1]+b[2]*y[i-1])+(1-Tran)*(b[3]+b[4]*y[i-1]);       
  u[i]  = y[i]-m[i];
  av[i] = (lambda^2)^0.5;
  i     = i+1;
endo;

retp(-0.5*ln(2*pi)-0.5*(u.^2));
endp;